local LrDialogs     = import 'LrDialogs'
local LrTasks       = import 'LrTasks'
local LrView        = import 'LrView'
local LrPathUtils   = import 'LrPathUtils'
local LrFileUtils   = import 'LrFileUtils'
local LrLogger      = import 'LrLogger'

local logger = LrLogger('WebPExport')
logger:enable('logfile')

local function isWin() return WIN_ENV end

local function quotePosix(s)
  if s == nil then return "''" end
  return "'" .. string.gsub(s, "'", "'\"'\"'") .. "'"
end

local function quoteWin(s)
  if s == nil then return '""' end
  s = string.gsub(s, '"', '""')
  return '"' .. s .. '"'
end


local function trim(s)
  return (s:gsub("^%s+", ""):gsub("%s+$", ""))
end

local function normalizeExePath(p)
  if not p then return "" end
  p = trim(p)
  if (#p >= 2) and ((p:sub(1,1) == '"' and p:sub(-1) == '"') or (p:sub(1,1) == "'" and p:sub(-1) == "'")) then
    p = p:sub(2, -2)
    p = trim(p)
  end
  return p
end

local function fileExists(path)
  return path and path ~= '' and LrFileUtils.exists(path)
end

local function replaceExtension(path, newExt)
  local dot = string.match(path, "^.*()%.")
  if dot then
    return string.sub(path, 1, dot - 1) .. "." .. newExt
  end
  return path .. "." .. newExt
end

local function buildCwebpArgs(settings, inputPath, outputPath)
  local args = {}

  if settings.webpPreset and settings.webpPreset ~= 'default' then
    table.insert(args, "-preset")
    table.insert(args, settings.webpPreset)
  end

  local mode = settings.webpMode or "lossy"
  local q = tonumber(settings.webpQuality) or 80
  q = math.max(0, math.min(100, q))

  if mode == "lossless" then
    table.insert(args, "-lossless")
    table.insert(args, "-q"); table.insert(args, string.format("%.0f", q))
    local z = tonumber(settings.webpZLevel) or 6
    z = math.max(0, math.min(9, z))
    table.insert(args, "-z"); table.insert(args, tostring(z))
  elseif mode == "near_lossless" then
    local nl = tonumber(settings.webpNearLossless) or 60
    nl = math.max(0, math.min(100, nl))
    table.insert(args, "-near_lossless"); table.insert(args, tostring(nl))
    table.insert(args, "-q"); table.insert(args, string.format("%.0f", q))
  else
    table.insert(args, "-q"); table.insert(args, string.format("%.0f", q))
  end

  local m = tonumber(settings.webpMethod) or 4
  m = math.max(0, math.min(6, m))
  table.insert(args, "-m"); table.insert(args, tostring(m))

  if settings.webpUseMt then table.insert(args, "-mt") end

  local md = settings.webpMetadata or "none"
  if md and md ~= "" then
    table.insert(args, "-metadata"); table.insert(args, md)
  end

  table.insert(args, "-quiet")

  if isWin() then
    table.insert(args, quoteWin(inputPath))
    table.insert(args, "-o"); table.insert(args, quoteWin(outputPath))
  else
    table.insert(args, quotePosix(inputPath))
    table.insert(args, "-o"); table.insert(args, quotePosix(outputPath))
  end

  return table.concat(args, " ")
end

local function buildCommand(exePath, argsString)
  exePath = normalizeExePath(exePath)

  if isWin() then
    -- Run the executable directly on Windows (avoids cmd.exe quoting / PATH issues)
    return '"' .. exePath .. '" ' .. argsString
  else
    return quotePosix(exePath) .. ' ' .. argsString
  end
end

local function validateExe(exePath)
  exePath = normalizeExePath(exePath)
  if exePath == "" then return false end

  local cmd
  if isWin() then
    cmd = '"' .. exePath .. '" -version'
  else
    cmd = quotePosix(exePath) .. ' -version'
  end

  local rc = LrTasks.execute(cmd)
  return rc == 0
end

local function resolveCwebp(settings)
  local candidates = {}

  if settings.cwebpPath and settings.cwebpPath ~= "" then
    table.insert(candidates, normalizeExePath(settings.cwebpPath))
  end

  -- Common locations (especially Homebrew, since GUI apps may not inherit shell PATH on macOS)
  if not isWin() then
    table.insert(candidates, "/opt/homebrew/bin/cwebp")  -- Apple Silicon Homebrew
    table.insert(candidates, "/usr/local/bin/cwebp")     -- Intel Homebrew
    table.insert(candidates, "/usr/bin/cwebp")
    table.insert(candidates, "/bin/cwebp")
  else
    table.insert(candidates, "C:\\Program Files\\WebP\\bin\\cwebp.exe")
    table.insert(candidates, "C:\\Program Files (x86)\\WebP\\bin\\cwebp.exe")
  end

  -- Plugin-local candidates
  local binDir = LrPathUtils.child(_PLUGIN.path, "bin")
  if isWin() then
    table.insert(candidates, LrPathUtils.child(binDir, "cwebp.exe"))
    table.insert(candidates, LrPathUtils.child(binDir, "windows\\cwebp.exe"))
    table.insert(candidates, "cwebp") -- PATH fallback
  else
    table.insert(candidates, LrPathUtils.child(binDir, "cwebp"))
    table.insert(candidates, LrPathUtils.child(binDir, "macos/cwebp"))
    table.insert(candidates, "cwebp") -- PATH fallback (works only if LR inherits PATH)
  end

  for _, exe in ipairs(candidates) do
    if exe == "cwebp" then
      local rc = LrTasks.execute(isWin() and 'C:\\Windows\\System32\\where.exe cwebp' or 'cwebp -version')
      if rc == 0 then return exe end
    else
      if fileExists(exe) and validateExe(exe) then
        return exe
      end
    end
  end

  return nil
end

local function showConfigError()
  LrDialogs.message(
    "WebP Exporter",
    "cwebp non trovato.\n\nmacOS:\n• Se installato con Homebrew, i path tipici sono /opt/homebrew/bin/cwebp (Apple Silicon) o /usr/local/bin/cwebp (Intel).\n• Imposta il percorso completo in Export > WebP (cwebp) > cwebp executable path.\n\nWindows:\n• Scarica le WebP utilities (libwebp) e individua cwebp.exe nella cartella bin.\n• Imposta il percorso completo in Export > WebP (cwebp) > cwebp executable path (es. C:\\Tools\\WebP\\bin\\cwebp.exe).\n\nAlternative:\n1) Metti cwebp nel PATH (non sempre visibile alle app GUI), oppure\n2) Copia cwebp dentro il plugin: LightroomWebPExport.lrplugin/bin/ (macOS: cwebp, Windows: cwebp.exe)."
  )
end

local WebP = {}
WebP.hideSections = { "fileSettings", "video" }

WebP.exportPresetFields = {
  { key = 'webpMode',           default = 'lossy' },
  { key = 'webpQuality',        default = 80 },
  { key = 'webpNearLossless',   default = 60 },
  { key = 'webpZLevel',         default = 6 },
  { key = 'webpMethod',         default = 4 },
  { key = 'webpPreset',         default = 'photo' },
  { key = 'webpMetadata',       default = 'icc,exif,xmp' },
  { key = 'webpUseMt',          default = true },
  { key = 'webpColorSpace',     default = 'sRGB' },
  { key = 'deleteIntermediate', default = true },
  { key = 'cwebpPath',          default = '' },
}

function WebP.startDialog(propertyTable)
  propertyTable.webpQuality = tonumber(propertyTable.webpQuality) or 80
  propertyTable.webpNearLossless = tonumber(propertyTable.webpNearLossless) or 60
  propertyTable.webpZLevel = tonumber(propertyTable.webpZLevel) or 6
  propertyTable.webpMethod = tonumber(propertyTable.webpMethod) or 4
end

function WebP.sectionsForBottomOfDialog(f, propertyTable)
  return {
    {
      title = "WebP (cwebp) Settings",
      synopsis = function(props)
        return string.format("%s, q=%s, preset=%s", tostring(props.webpMode), tostring(props.webpQuality), tostring(props.webpPreset))
      end,
      f:row {
        spacing = f:control_spacing(),
        f:static_text { title = "Mode", width = 120 },
        f:popup_menu {
          value = LrView.bind('webpMode'),
          items = {
            { title="Lossy", value="lossy" },
            { title="Lossless", value="lossless" },
            { title="Near-lossless", value="near_lossless" },
          },
        },
      },
      f:row {
        spacing = f:control_spacing(),
        f:static_text { title = "Quality (0–100)", width = 120 },
        f:edit_field { value = LrView.bind('webpQuality'), width_in_chars = 6 },
        f:static_text { title = " (lossy + lossless)" },
      },
      f:row {
        spacing = f:control_spacing(),
        f:static_text { title = "Near-lossless (0–100)", width = 120 },
        f:edit_field {
          value = LrView.bind('webpNearLossless'),
          width_in_chars = 6,
          enabled = LrView.bind { keys = { 'webpMode' }, transform = function(v) return v == "near_lossless" end },
        },
      },
      f:row {
        spacing = f:control_spacing(),
        f:static_text { title = "Lossless level (0–9)", width = 120 },
        f:edit_field {
          value = LrView.bind('webpZLevel'),
          width_in_chars = 6,
          enabled = LrView.bind { keys = { 'webpMode' }, transform = function(v) return v == "lossless" end },
        },
      },
      f:row {
        spacing = f:control_spacing(),
        f:static_text { title = "Preset", width = 120 },
        f:popup_menu {
          value = LrView.bind('webpPreset'),
          items = {
            { title="photo", value="photo" },
            { title="picture", value="picture" },
            { title="drawing", value="drawing" },
            { title="icon", value="icon" },
            { title="text", value="text" },
            { title="default", value="default" },
          },
        },
      },
      f:row {
        spacing = f:control_spacing(),
        f:static_text { title = "Method (0–6)", width = 120 },
        f:edit_field { value = LrView.bind('webpMethod'), width_in_chars = 6 },
        f:checkbox { title="Multi-thread (-mt)", value = LrView.bind('webpUseMt') },
      },
      f:row {
        spacing = f:control_spacing(),
        f:static_text { title = "Metadata", width = 120 },
        f:popup_menu {
          value = LrView.bind('webpMetadata'),
          items = {
            { title="none", value="none" },
            { title="icc", value="icc" },
            { title="exif", value="exif" },
            { title="xmp", value="xmp" },
            { title="icc,exif,xmp", value="icc,exif,xmp" },
            { title="all", value="all" },
          },
        },
      },
      f:row {
        spacing = f:control_spacing(),
        f:static_text { title = "Render color space", width = 120 },
        f:popup_menu {
          value = LrView.bind('webpColorSpace'),
          items = {
            { title="sRGB", value="sRGB" },
            { title="Adobe RGB", value="AdobeRGB" },
            { title="ProPhoto RGB", value="ProPhotoRGB" },
          },
        },
      },
      f:row { spacing = f:control_spacing(), f:checkbox { title="Delete intermediate TIFF", value = LrView.bind('deleteIntermediate') } },
      f:spacer { height = 8 },
      f:row {
        spacing = f:control_spacing(),
        f:static_text { title = "cwebp executable path", width = 120 },
        f:edit_field {
          value = LrView.bind('cwebpPath'),
          width_in_chars = 42,
          placeholder_string = isWin() and "C:\\path\\to\\cwebp.exe" or "/opt/homebrew/bin/cwebp  (or /usr/local/bin/cwebp)",
        },
        f:push_button {
          title = "Browse…",
          action = function()
            local res = LrDialogs.runOpenPanel {
              title = "Select cwebp executable",
              canChooseFiles = true,
              canChooseDirectories = false,
              allowsMultipleSelection = false,
            }
            if res and res[1] then
              propertyTable.cwebpPath = res[1]
            end
          end,
        },
      },
    },
  }
end

function WebP.updateExportSettings(exportSettings)
  exportSettings.LR_format = "TIFF"
  exportSettings.LR_export_bitDepth = 8
  exportSettings.LR_tiff_compressionMethod = "none"
  exportSettings.LR_export_addToCatalog = false

  if exportSettings.webpColorSpace and exportSettings.webpColorSpace ~= "" then
    exportSettings.LR_export_colorSpace = exportSettings.webpColorSpace
  else
    exportSettings.LR_export_colorSpace = "sRGB"
  end
end

function WebP.processRenderedPhotos(functionContext, exportContext)
  local exportSettings = exportContext.propertyTable
  local exportSession = exportContext.exportSession
  local nRenditions = exportSession:countRenditions()

  local exe = resolveCwebp(exportSettings)
  if not exe then
    showConfigError()
    return
  end

  local progressScope = exportContext:configureProgress {
    title = "Exporting WebP (cwebp)",
    caption = "Rendering + converting…",
  }
  progressScope:setCancelable(true)

  local failures = 0
  local i = 0

  for _, rendition in exportContext:renditions { stopIfCanceled = true } do
    i = i + 1
    progressScope:setPortionComplete(i - 1, nRenditions)

    if progressScope:isCanceled() then
      rendition:renditionIsDone(false, "Canceled")
      break
    end

    local ok, pathOrMessage = rendition:waitForRender()
    if not ok then
      failures = failures + 1
      rendition:renditionIsDone(false, tostring(pathOrMessage))
    else
      local renderedPath = rendition.destinationPath or pathOrMessage
      local webpPath = replaceExtension(renderedPath, "webp")

      local args = buildCwebpArgs(exportSettings, renderedPath, webpPath)
      local cmd = (exe == "cwebp") and ("cwebp " .. args) or buildCommand(exe, args)

      logger:trace("CMD: " .. cmd)
      local rc = LrTasks.execute(cmd)

      if rc ~= 0 or not LrFileUtils.exists(webpPath) then
        failures = failures + 1
        logger:error(string.format("cwebp failed rc=%s (in=%s)", tostring(rc), tostring(renderedPath)))
        rendition:renditionIsDone(false, string.format("cwebp failed (rc=%s). Check plugin log.", tostring(rc)))
      else
        if exportSettings.deleteIntermediate then
          pcall(function() LrFileUtils.delete(renderedPath) end)
        end
        rendition:renditionIsDone(true)
      end
    end
  end

  progressScope:done()

  if failures > 0 then
    LrDialogs.message("WebP Exporter", string.format("Export completato con %d errori. Controlla il log del plugin.", failures), "warning")
  end
end

return WebP
